<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureCategory;

/**
 * CategoryController
 * Beheert furniture categorieën CRUD operaties voor VPS2
 */
class CategoryController extends Controller
{
    private FurnitureCategory $categoryModel;

    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
        $this->categoryModel = new FurnitureCategory();
    }

    /**
     * Overzicht van alle categorieën
     */
    public function index(): void
    {
        $categories = $this->categoryModel->getAllWithProductCount();

        $this->view('categories.index', [
            'user' => $this->user(),
            'title' => 'Categorieën',
            'categories' => $categories
        ]);
    }

    /**
     * Nieuw categorie formulier
     */
    public function create(): void
    {
        $this->view('categories.create', [
            'user' => $this->user(),
            'title' => 'Nieuwe Categorie'
        ]);
    }

    /**
     * Categorie opslaan
     */
    public function store(): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        // Input ophalen
        $data = [
            'name' => $_POST['name'] ?? '',
            'description' => $_POST['description'] ?? null,
            'sort_order' => $_POST['sort_order'] ?? 0,
            'status' => $_POST['status'] ?? 'active'
        ];

        // Validatie
        $errors = $this->validateCategory($data);

        if (!empty($errors)) {
            foreach ($errors as $field => $message) {
                $this->flash('error', $message);
            }
            $this->back();
            return;
        }

        // Slug genereren
        $data['slug'] = $this->categoryModel->generateSlug($data['name']);

        // Categorie aanmaken
        $categoryId = $this->categoryModel->create($data);

        if ($categoryId) {
            $this->flash('success', 'Categorie succesvol toegevoegd.');
            $this->redirect('/categories');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het aanmaken van de categorie.');
            $this->back();
        }
    }

    /**
     * Categorie bewerken formulier
     */
    public function edit(string $slug): void
    {
        $category = $this->categoryModel->findBySlug($slug);

        if (!$category) {
            $this->flash('error', 'Categorie niet gevonden.');
            $this->redirect('/categories');
            return;
        }

        $this->view('categories.edit', [
            'user' => $this->user(),
            'title' => 'Categorie Bewerken',
            'category' => $category
        ]);
    }

    /**
     * Categorie updaten
     */
    public function update(string $slug): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        $category = $this->categoryModel->findBySlug($slug);

        if (!$category) {
            $this->flash('error', 'Categorie niet gevonden.');
            $this->redirect('/categories');
            return;
        }

        $id = $category['id'];

        // Input ophalen
        $data = [
            'name' => $_POST['name'] ?? '',
            'description' => $_POST['description'] ?? null,
            'sort_order' => $_POST['sort_order'] ?? 0,
            'status' => $_POST['status'] ?? 'active'
        ];

        // Validatie
        $errors = $this->validateCategory($data, $id);

        if (!empty($errors)) {
            foreach ($errors as $field => $message) {
                $this->flash('error', $message);
            }
            $this->back();
            return;
        }

        // Slug genereren als naam is veranderd
        if ($data['name'] !== $category['name']) {
            $data['slug'] = $this->categoryModel->generateSlug($data['name']);
        }

        if ($this->categoryModel->update($id, $data)) {
            $this->flash('success', 'Categorie succesvol bijgewerkt.');
            $this->redirect('/categories');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het bijwerken van de categorie.');
            $this->back();
        }
    }

    /**
     * Categorie verwijderen
     */
    public function delete(string $slug): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->redirect('/categories');
            return;
        }

        $category = $this->categoryModel->findBySlug($slug);

        if (!$category) {
            $this->flash('error', 'Categorie niet gevonden.');
            $this->redirect('/categories');
            return;
        }

        $id = $category['id'];

        // Check of er producten gekoppeld zijn aan deze categorie
        $stmt = $this->db->getConnection()->prepare(
            "SELECT COUNT(*) as count FROM furniture_products WHERE category = ?"
        );
        $stmt->execute([$category['name']]);
        $result = $stmt->fetch();

        if ($result['count'] > 0) {
            $this->flash('error', "Kan categorie niet verwijderen: er zijn nog {$result['count']} product(en) gekoppeld aan deze categorie.");
            $this->redirect('/categories');
            return;
        }

        if ($this->categoryModel->delete($id)) {
            $this->flash('success', 'Categorie succesvol verwijderd.');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het verwijderen van de categorie.');
        }

        $this->redirect('/categories');
    }

    /**
     * Valideer categorie data
     */
    private function validateCategory(array $data, ?int $excludeId = null): array
    {
        $errors = [];

        if (empty($data['name'])) {
            $errors['name'] = 'Categorienaam is verplicht.';
        } else {
            // Check of naam al bestaat (bij andere categorie)
            $existing = $this->categoryModel->where('name', $data['name']);
            if ($existing && (!$excludeId || $existing['id'] != $excludeId)) {
                $errors['name'] = 'Deze categorienaam bestaat al.';
            }
        }

        if (!empty($data['sort_order']) && !is_numeric($data['sort_order'])) {
            $errors['sort_order'] = 'Sorteervolgorde moet een getal zijn.';
        }

        if (!in_array($data['status'], ['active', 'inactive'])) {
            $errors['status'] = 'Ongeldige status.';
        }

        return $errors;
    }
}
