<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureOrder;
use App\Models\FurnitureProduct;
use App\Models\ApiCredential;

/**
 * DashboardController
 * Admin dashboard voor VPS2 monitoring en beheer
 */
class DashboardController extends Controller
{
    private FurnitureOrder $orderModel;
    private FurnitureProduct $productModel;
    private ApiCredential $apiCredentialModel;

    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
        $this->orderModel = new FurnitureOrder();
        $this->productModel = new FurnitureProduct();
        $this->apiCredentialModel = new ApiCredential();
    }

    /**
     * Dashboard home
     */
    public function index(): void
    {
        // Statistieken
        $orders = $this->orderModel->all();
        
        $stats = [
            'total_orders' => count($orders),
            'pending_orders' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'pending')),
            'paid_orders' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'paid')),
            'failed_orders' => count(array_filter($orders, fn($o) => $o['payment_status'] === 'failed')),
            'pending_webhooks' => count(array_filter($orders, fn($o) => !$o['webhook_sent'] && in_array($o['payment_status'], ['paid', 'failed', 'cancelled']))),
            'total_products' => count($this->productModel->all()),
            'recent_orders' => array_slice($orders, 0, 10)
        ];

        $this->view('dashboard.index', [
            'user' => $this->user(),
            'title' => 'Dashboard',
            'stats' => $stats
        ]);
    }

    /**
     * Orders overzicht (GEPAGINEERD)
     */
    public function orders(): void
    {
        // Pagination parameters
        $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
        $perPage = 25; // Orders per pagina
        $offset = ($page - 1) * $perPage;

        // Haal gepagineerde orders op
        $orders = $this->orderModel->getAllWithDetails($perPage, $offset);
        
        // Tel totaal aantal orders
        $totalOrders = $this->orderModel->countAll();
        $totalPages = ceil($totalOrders / $perPage);

        // Statistieken (efficient met GROUP BY)
        $statusCounts = $this->orderModel->countByStatus();

        $this->view('dashboard.orders', [
            'user' => $this->user(),
            'title' => 'Orders',
            'orders' => $orders,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total_orders' => $totalOrders,
                'total_pages' => $totalPages,
                'has_prev' => $page > 1,
                'has_next' => $page < $totalPages
            ],
            'stats' => $statusCounts
        ]);
    }

    /**
     * Order details
     */
    public function viewOrder(string $id): void
    {
        $order = $this->orderModel->findByFurnitureOrderId($id);

        if (!$order) {
            $this->flash('error', 'Order niet gevonden.');
            $this->redirect('/dashboard/orders');
            return;
        }

        $this->view('dashboard.order-view', [
            'user' => $this->user(),
            'title' => 'Order Details',
            'order' => $order
        ]);
    }

    /**
     * Instellingen
     */
    public function settings(): void
    {
        $credentials = $this->apiCredentialModel->getActiveCredentials();
        $products = $this->productModel->getActive();

        $this->view('dashboard.settings', [
            'user' => $this->user(),
            'title' => 'Instellingen',
            'credentials' => $credentials,
            'products' => $products
        ]);
    }

    /**
     * Instellingen opslaan
     */
    public function updateSettings(): void
    {
        // Hier kun je instellingen opslaan
        $this->flash('success', 'Instellingen succesvol opgeslagen.');
        $this->redirect('/dashboard/settings');
    }

    /**
     * Genereer nieuwe API credentials
     */
    public function generateCredentials(): void
    {
        $credentials = $this->apiCredentialModel->getActiveCredentials();
        
        if (!$credentials) {
            // Maak nieuwe credentials aan
            $id = $this->apiCredentialModel->createWithGeneratedCredentials('VPS1 API Key');
            if ($id) {
                $this->flash('success', 'Nieuwe API credentials succesvol gegenereerd!');
            } else {
                $this->flash('error', 'Fout bij aanmaken nieuwe credentials.');
            }
        } else {
            // Update bestaande credentials
            if ($this->apiCredentialModel->generateNewCredentials($credentials['id'])) {
                $this->flash('success', 'Nieuwe API credentials succesvol gegenereerd! Kopieer de nieuwe waarden naar VPS1.');
            } else {
                $this->flash('error', 'Fout bij genereren nieuwe credentials.');
            }
        }
        
        $this->redirect('/dashboard/settings');
    }

    /**
     * Check VPS1 connectie status
     * GET /dashboard/settings/check-connection
     * Test BEIDE richtingen:
     * 1. VPS2 → VPS1 (webhook): Test of VPS1 onze webhooks accepteert
     * 2. VPS1 → VPS2 (API): Test of VPS1's API key bij ons klopt
     */
    public function checkConnection(): void
    {
        // Zet JSON header direct VOOR alles
        header('Content-Type: application/json');
        
        // Start output buffering om te voorkomen dat PHP errors/warnings de JSON response verstoren
        while (ob_get_level()) {
            ob_end_clean();
        }
        ob_start();
        
        // Zet error reporting uit voor deze functie
        $oldErrorReporting = error_reporting(0);
        $oldDisplayErrors = ini_set('display_errors', 0);
        
        try {
            // Laad config helper functie als die niet bestaat
            if (!function_exists('config')) {
                require_once __DIR__ . '/../../core/helpers.php';
            }
            
            $webhookUrl = config('app.vps1_webhook_url');
            $credentials = $this->apiCredentialModel->getActiveCredentials();
        } catch (\Throwable $e) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200); // Zet status code expliciet
            echo json_encode([
                'connected' => false,
                'message' => 'Fout bij ophalen configuratie: ' . $e->getMessage()
            ]);
            exit;
        }
        
        if (!$webhookUrl) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (Webhook URL niet geconfigureerd)'
            ]);
            exit;
        }

        if (!$credentials || !isset($credentials['webhook_secret']) || !$credentials['webhook_secret']) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (Credentials ontbreken)'
            ]);
            exit;
        }

        // === TEST 1: VPS2 → VPS1 Webhook Connectie ===
        // Test of VPS1 bereikbaar is
        // URL-encode de URL om spaties te escapen (Dashboard 1 heeft spatie in naam)
        $encodedWebhookUrl = str_replace(' ', '%20', $webhookUrl);
        $ch = \curl_init($encodedWebhookUrl);
        \curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_NOBODY => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ]);

        $success = \curl_exec($ch);
        $httpCode = \curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = \curl_error($ch);
        \curl_close($ch);

        if ($success === false || !empty($error) || $httpCode >= 500) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (Webhook niet bereikbaar: ' . ($error ?: "HTTP $httpCode") . ')'
            ]);
            exit;
        }

        // Test webhook credentials
        $testPayload = [
            'event' => 'connection.test',
            'furniture_order_id' => 'TEST-' . time(),
            'external_order_id' => 'TEST',
            'payment_status' => 'pending',
            'amount' => 0,
            'currency' => 'EUR',
            'test' => true
        ];

        $payloadJson = json_encode($testPayload);
        $signature = hash_hmac('sha256', $payloadJson, $credentials['webhook_secret']);

        // Gebruik URL-encoded versie voor curl (Dashboard 1 heeft spatie in naam)
        $encodedWebhookUrl = str_replace(' ', '%20', $webhookUrl);
        $ch = \curl_init($encodedWebhookUrl);
        \curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payloadJson,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-Signature: ' . $signature
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ]);

        $response = \curl_exec($ch);
        $webhookHttpCode = \curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $webhookError = \curl_error($ch);
        \curl_close($ch);

        // Als webhook 401 geeft, betekent het dat de webhook secret niet klopt
        if ($webhookHttpCode === 401) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (Webhook secret ongeldig)'
            ]);
            exit;
        }
        
        // Als webhook niet 200 geeft, is er iets mis
        if ($webhookHttpCode !== 200) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            $errorMsg = $webhookError ?: "HTTP $webhookHttpCode";
            if ($webhookHttpCode === 0) {
                $errorMsg = 'Verbinding gefaald (timeout of DNS error)';
            } elseif ($webhookHttpCode === 404) {
                $errorMsg = 'Endpoint niet gevonden (404)';
            } elseif ($webhookHttpCode >= 500) {
                $errorMsg = 'Server error (' . $webhookHttpCode . ')';
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (TEST 1: Webhook test gefaald: ' . $errorMsg . ')'
            ]);
            exit;
        }

        // === TEST 2: VPS1 → VPS2 API Connectie ===
        // Test of VPS1's credentials werken door VPS2's eigen API aan te roepen
        // (zoals VPS1 dat zou doen)
        
        $vps2BaseUrl = config('app.url');
        if (!$vps2BaseUrl) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (VPS2 base URL niet geconfigureerd)'
            ]);
            exit;
        }
        
        $testUrl = $vps2BaseUrl . '/api/v1/products';
        
        // Test VPS2's API met de credentials die VPS1 gebruikt
        // Dit simuleert wat VPS1 doet wanneer het naar VPS2 callt
        if (!function_exists('curl_init')) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (cURL niet beschikbaar)'
            ]);
            exit;
        }
        
        $ch = \curl_init($testUrl);
        if ($ch === false) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (Kon cURL niet initialiseren)'
            ]);
            exit;
        }
        
        // Controleer of API key bestaat
        if (empty($credentials['api_key'])) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (API key ontbreekt in credentials)'
            ]);
            exit;
        }
        
        \curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 5,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_HTTPHEADER => [
                'X-API-Key: ' . $credentials['api_key'],
                'Content-Type: application/json'
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ]);
        
        $testResponse = \curl_exec($ch);
        $testHttpCode = \curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $testError = \curl_error($ch);
        \curl_close($ch);
        
        if ($testHttpCode !== 200 || !$testResponse) {
            // API test gefaald
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            $errorMsg = $testError ?: "HTTP $testHttpCode";
            if ($testHttpCode === 401) {
                $errorMsg = 'API key ongeldig (HTTP 401)';
            } elseif ($testHttpCode === 0) {
                $errorMsg = 'Verbinding gefaald (timeout of DNS error)';
            }
            
            // Debug info toevoegen (alleen in development)
            $debugInfo = '';
            if (config('app.debug', false)) {
                $debugInfo = ' | URL: ' . $testUrl . ' | Key: ' . substr($credentials['api_key'] ?? '', 0, 10) . '...';
            }
            
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'Niet verbonden (VPS1 → VPS2 API test gefaald: ' . $errorMsg . $debugInfo . ')'
            ]);
            exit;
        }
        
        // Parse response
        $testResult = json_decode($testResponse, true);
        if (!$testResult || !isset($testResult['success']) || !$testResult['success']) {
            ob_end_clean();
            error_reporting($oldErrorReporting);
            if ($oldDisplayErrors !== false) {
                ini_set('display_errors', $oldDisplayErrors);
            }
            $errorMsg = $testResult['error'] ?? 'Onbekende fout';
            $responsePreview = substr($testResponse, 0, 200);
            http_response_code(200);
            echo json_encode([
                'connected' => false,
                'message' => 'VPS1 → VPS2 API: Test gefaald (' . $errorMsg . ') | Response: ' . $responsePreview
            ]);
            exit;
        }
        
        // Alle tests geslaagd!
        ob_end_clean();
        error_reporting($oldErrorReporting);
        if ($oldDisplayErrors !== false) {
            ini_set('display_errors', $oldDisplayErrors);
        }
        $productCount = $testResult['count'] ?? count($testResult['products'] ?? []);
        http_response_code(200);
        echo json_encode([
            'connected' => true,
            'message' => 'Verbonden • ' . $productCount . ' ' . ($productCount === 1 ? 'product' : 'producten')
        ]);
        exit;
    }
}

