<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureOrder;
use App\Models\ApiCredential;
use App\Models\FurnitureProduct;
use App\Services\WebhookService;

/**
 * OrderApiController
 * API endpoints voor VPS1 om orders te maken en status te checken
 */
class OrderApiController extends Controller
{
    private FurnitureOrder $orderModel;
    private ApiCredential $apiCredentialModel;
    private FurnitureProduct $productModel;
    private WebhookService $webhookService;

    public function __construct()
    {
        parent::__construct();
        $this->orderModel = new FurnitureOrder();
        $this->apiCredentialModel = new ApiCredential();
        $this->productModel = new FurnitureProduct();
        $this->webhookService = new WebhookService();
    }

    /**
     * POST /api/v1/orders/create
     * Ontvangt order van VPS1 en maakt furniture order aan
     */
    public function createOrder(): void
    {
        // Verifieer API authenticatie
        if (!$this->verifyApiAuth()) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'Invalid API key or signature'
            ], 401);
            return;
        }

        // Log API call
        $this->logApiCall('POST', '/api/v1/orders/create');

        // Haal request data op
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$data) {
            $this->json([
                'success' => false,
                'error' => 'Invalid JSON',
                'message' => 'Request body must be valid JSON'
            ], 400);
            return;
        }

        // Valideer required fields
        $required = ['external_order_id', 'furniture_product_id', 'furniture_product_name', 'price'];
        foreach ($required as $field) {
            if (!isset($data[$field])) {
                $this->json([
                    'success' => false,
                    'error' => 'Missing required field',
                    'message' => "Field '{$field}' is required"
                ], 400);
                return;
            }
        }

        // Check of dit een test order is (voor connection testing)
        $isTestOrder = isset($data['external_order_id']) && strpos($data['external_order_id'], 'TEST-') === 0;
        
        // Als het een test order is, geef alleen authenticatie bevestiging terug zonder order aan te maken
        if ($isTestOrder) {
            // Log success (authenticatie werkt) - geen error message want het is een succesvolle test
            $this->logApiCall('POST', '/api/v1/orders/create', 200);
            
            // Return success response zonder order aan te maken
            $this->json([
                'success' => true,
                'test' => true,
                'message' => 'Authentication successful - test order not created',
                'furniture_order_id' => 'TEST-' . time(),
                'payment_url' => config('app.url') . '/checkout/test',
                'expires_at' => date('c', strtotime('+1 hour'))
            ], 200);
            return;
        }

        // Valideer dat product bestaat (optioneel - kan ook alleen naam gebruiken)
        $product = $this->productModel->findByProductId($data['furniture_product_id']);
        if ($product && $product['price'] != $data['price']) {
            // Waarschuwing: prijs komt niet overeen
            error_log("Price mismatch: Product {$data['furniture_product_id']} heeft prijs {$product['price']}, maar order heeft {$data['price']}");
        }

        try {
            // Maak furniture order aan
            $furnitureOrderId = $this->orderModel->createOrder($data);

            if (!$furnitureOrderId) {
                $this->json([
                    'success' => false,
                    'error' => 'Database error',
                    'message' => 'Failed to create order'
                ], 500);
                return;
            }

            // Haal order op voor response
            $order = $this->orderModel->findByFurnitureOrderId($furnitureOrderId);

            // Genereer payment URL (placeholder - moet gekoppeld worden aan payment provider)
            $paymentUrl = $this->generatePaymentUrl($furnitureOrderId);

            // Update order met payment URL
            $this->orderModel->updatePaymentStatus($furnitureOrderId, 'pending', [
                'payment_url' => $paymentUrl,
                'payment_url_expires_at' => date('Y-m-d H:i:s', strtotime('+1 hour'))
            ]);

            // Stuur webhook naar VPS1 dat order is aangemaakt (met status pending)
            try {
                $this->webhookService->sendOrderCreated($furnitureOrderId);
            } catch (\Exception $e) {
                // Log error maar stop niet de response
                error_log("WebhookService: Fout bij versturen order.created webhook: " . $e->getMessage());
            }

            // Log success
            $this->logApiCall('POST', '/api/v1/orders/create', 200);

            // Return success response
            $this->json([
                'success' => true,
                'furniture_order_id' => $furnitureOrderId,
                'payment_url' => $paymentUrl,
                'expires_at' => date('c', strtotime('+1 hour'))
            ], 201);

        } catch (\Exception $e) {
            $this->logApiCall('POST', '/api/v1/orders/create', 500, $e->getMessage());
            
            $this->json([
                'success' => false,
                'error' => 'Internal server error',
                'message' => 'An error occurred while creating the order'
            ], 500);
        }
    }

    /**
     * GET /api/v1/orders/{id}/status
     * Returnt payment status van furniture order
     */
    public function getOrderStatus(string $id): void
    {
        // Verifieer API authenticatie
        if (!$this->verifyApiAuth()) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'Invalid API key or signature'
            ], 401);
            return;
        }

        // Log API call
        $this->logApiCall('GET', "/api/v1/orders/{$id}/status");

        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($id);

        if (!$order) {
            $this->json([
                'success' => false,
                'error' => 'Not found',
                'message' => 'Order not found'
            ], 404);
            return;
        }

        // Return order status
        $this->json([
            'success' => true,
            'furniture_order_id' => $order['furniture_order_id'],
            'external_order_id' => $order['external_order_id'],
            'payment_status' => $order['payment_status'],
            'paid_at' => $order['paid_at'],
            'transaction_id' => $order['transaction_id']
        ], 200);
    }

    /**
     * POST /api/v1/orders/{id}/payment-status
     * Update payment status (bijv. van payment provider webhook)
     * Stuurt automatisch webhook naar VPS1
     */
    public function updatePaymentStatus(string $id): void
    {
        // Verifieer API authenticatie
        if (!$this->verifyApiAuth()) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'Invalid API key or signature'
            ], 401);
            return;
        }

        // Haal request data op
        $data = json_decode(file_get_contents('php://input'), true);

        if (!$data || !isset($data['payment_status'])) {
            $this->json([
                'success' => false,
                'error' => 'Invalid request',
                'message' => 'payment_status is required'
            ], 400);
            return;
        }

        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($id);

        if (!$order) {
            $this->json([
                'success' => false,
                'error' => 'Not found',
                'message' => 'Order not found'
            ], 404);
            return;
        }

        // Update payment status
        $updateData = [
            'transaction_id' => $data['transaction_id'] ?? $order['transaction_id'],
        ];

        if ($data['payment_status'] === 'paid') {
            $updateData['paid_at'] = $data['paid_at'] ?? date('Y-m-d H:i:s');
        } elseif ($data['payment_status'] === 'failed') {
            $updateData['failed_at'] = $data['failed_at'] ?? date('Y-m-d H:i:s');
            $updateData['failure_reason'] = $data['failure_reason'] ?? null;
        }

        $this->orderModel->updatePaymentStatus($id, $data['payment_status'], $updateData);

        // Stuur automatisch webhook naar VPS1
        $webhookSent = $this->webhookService->sendPaymentStatusUpdate(
            $id,
            $data['payment_status'],
            $updateData
        );

        // Log API call
        $this->logApiCall('POST', "/api/v1/orders/{$id}/payment-status", 200);

        $this->json([
            'success' => true,
            'message' => 'Payment status updated',
            'webhook_sent' => $webhookSent
        ], 200);
    }

    /**
     * Verifieer API authenticatie (API key + HMAC signature)
     */
    private function verifyApiAuth(): bool
    {
        // Haal API key op uit header
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? null;
        $signature = $_SERVER['HTTP_X_SIGNATURE'] ?? null;

        if (!$apiKey || !$signature) {
            return false;
        }

        // Haal credentials op uit database
        $credentials = $this->apiCredentialModel->findByApiKey($apiKey);

        if (!$credentials || !$credentials['is_active']) {
            return false;
        }

        // Haal secret op (niet encrypted in deze versie)
        $secret = $credentials['api_secret'];

        // Haal request body op
        $requestBody = file_get_contents('php://input');
        
        // Bereken HMAC signature
        $calculatedSignature = hash_hmac('sha256', $requestBody, $secret);

        // Verifieer signature
        return hash_equals($signature, $calculatedSignature);
    }

    /**
     * Genereer payment URL (placeholder - moet gekoppeld worden aan payment provider)
     */
    private function generatePaymentUrl(string $furnitureOrderId): string
    {
        $config = require __DIR__ . '/../../config/app.php';
        $baseUrl = rtrim($config['url'], '/');
        
        // Placeholder payment URL - moet vervangen worden door echte payment provider URL
        return "{$baseUrl}/checkout/{$furnitureOrderId}";
    }

    /**
     * Log API call naar database
     */
    private function logApiCall(string $method, string $endpoint, int $statusCode = 200, ?string $error = null): void
    {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? null;
        $apiKeyHash = $apiKey ? hash('sha256', $apiKey) : null;
        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $requestBody = file_get_contents('php://input');
        
        $this->db->execute(
            "INSERT INTO api_access_log 
             (api_key_hash, endpoint, ip_address, request_method, request_payload, response_status, error_message, created_at)
             VALUES (?, ?, ?, ?, ?, ?, ?, NOW())",
            [
                $apiKeyHash,
                $endpoint,
                $ipAddress,
                $method,
                $requestBody ?: null,
                $statusCode,
                $error
            ]
        );
    }
}

