<?php
namespace App\Controllers;

use Core\Controller;
use App\Models\FurnitureProduct;
use App\Models\ApiCredential;

/**
 * ProductApiController
 * API endpoints voor VPS1 om furniture producten op te halen
 */
class ProductApiController extends Controller
{
    private FurnitureProduct $productModel;
    private ApiCredential $apiCredentialModel;

    public function __construct()
    {
        parent::__construct();
        $this->productModel = new FurnitureProduct();
        $this->apiCredentialModel = new ApiCredential();
    }

    /**
     * GET /api/v1/products
     * Haal alle furniture producten op (voor dropdown in VPS1)
     */
    public function index(): void
    {
        // Verifieer API authenticatie (alleen API key voor GET requests)
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
        
        if (empty($apiKey)) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'API key required'
            ], 401);
            return;
        }

        $credentials = $this->apiCredentialModel->findByApiKey($apiKey);
        
        if (!$credentials || !$credentials['is_active']) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'Invalid API key'
            ], 401);
            return;
        }

        // Haal alle actieve producten op
        $products = $this->productModel->getActive();

        // Haal base URL op EEN KEER (buiten de closure)
        // Gebruik config() helper i.p.v. require om pad issues te voorkomen
        $baseUrl = config('app.url', 'http://localhost');
        
        // Detecteer protocol en host als fallback
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $scriptPath = $_SERVER['SCRIPT_NAME'] ?? '/index.php';
        $publicPath = dirname($scriptPath);
        
        // Format voor VPS1 (gebruik id ipv product_id voor frontend compatibility)
        $formatted = array_map(function($product) use ($baseUrl, $scheme, $host, $publicPath) {
            // Database heeft is_active (BOOLEAN), niet status
            // Convert is_active (1/0) naar status ('active'/'inactive')
            $isActive = isset($product['is_active']) ? (bool)$product['is_active'] : true;
            
            // Build image URL (relatief pad vanaf public folder)
            $imageUrl = null;
            if (!empty($product['image_url'])) {
                // Als het al een volledige URL is, gebruik die
                if (strpos($product['image_url'], 'http') === 0) {
                    $imageUrl = $product['image_url'];
                } else {
                    // Bouw volledige URL (image_url is relatief vanaf public folder)
                    // Bijv: uploads/products/image.jpg -> http://localhost/Dashboard%202/public/uploads/products/image.jpg
                    // Probeer eerst baseUrl, anders bouw op basis van huidige request
                    if ($baseUrl && $baseUrl !== 'http://localhost') {
                        $imageUrl = rtrim($baseUrl, '/') . '/' . ltrim($product['image_url'], '/');
                    } else {
                        // Fallback: bouw URL op basis van huidige request
                        $imageUrl = $scheme . '://' . $host . $publicPath . '/' . ltrim($product['image_url'], '/');
                    }
                }
            }
            
            return [
                'id' => $product['product_id'], // Frontend verwacht 'id'
                'product_id' => $product['product_id'], // Legacy compatibility
                'name' => $product['name'],
                'sku' => $product['sku'] ?? '',
                'price' => (float) $product['price'],
                'currency' => $product['currency'] ?? 'EUR',
                'description' => $product['description'] ?? '',
                'stock' => (int) ($product['stock'] ?? 0), // Database heeft geen stock kolom, altijd 0
                'status' => $isActive ? 'active' : 'inactive',
                'image_url' => $imageUrl
            ];
        }, $products);

        $this->json([
            'success' => true,
            'products' => $formatted,
            'count' => count($formatted)
        ]);
    }

    /**
     * GET /api/v1/products/{id}
     * Haal specifiek furniture product op
     */
    public function show(string $id): void
    {
        // Verifieer API authenticatie (alleen API key voor GET requests)
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
        
        if (empty($apiKey)) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'API key required'
            ], 401);
            return;
        }

        $credentials = $this->apiCredentialModel->findByApiKey($apiKey);
        
        if (!$credentials || !$credentials['is_active']) {
            $this->json([
                'success' => false,
                'error' => 'Unauthorized',
                'message' => 'Invalid API key'
            ], 401);
            return;
        }

        $product = $this->productModel->findByProductId($id);

        if (!$product) {
            $this->json([
                'success' => false,
                'error' => 'Not found',
                'message' => 'Product not found'
            ], 404);
            return;
        }

        $this->json([
            'success' => true,
            'product' => [
                'product_id' => $product['product_id'],
                'name' => $product['name'],
                'sku' => $product['sku'],
                'price' => (float) $product['price'],
                'currency' => $product['currency'],
                'description' => $product['description']
            ]
        ]);
    }

    /**
     * Verifieer API authenticatie
     */
    private function verifyApiAuth(): bool
    {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? '';
        $signature = $_SERVER['HTTP_X_SIGNATURE'] ?? '';

        if (empty($apiKey)) {
            return false;
        }

        $credentials = $this->apiCredentialModel->findByApiKey($apiKey);

        if (!$credentials || !$credentials['is_active']) {
            return false;
        }

        // Verifieer HMAC signature
        $requestBody = file_get_contents('php://input');
        $expectedSignature = hash_hmac('sha256', $requestBody, $credentials['api_secret']);

        return hash_equals($expectedSignature, $signature);
    }
}

