<?php
namespace App\Controllers;

use Core\Controller;
use Core\FileUpload;
use App\Models\FurnitureProduct;
use App\Models\FurnitureProductImage;

/**
 * ProductController
 * Beheert furniture producten CRUD operaties voor VPS2
 */
class ProductController extends Controller
{
    private FurnitureProduct $productModel;
    private FurnitureProductImage $productImageModel;

    public function __construct()
    {
        parent::__construct();
        $this->requireAuth();
        $this->productModel = new FurnitureProduct();
        $this->productImageModel = new FurnitureProductImage();
    }

    /**
     * Overzicht van alle producten
     */
    public function index(): void
    {
        $products = $this->productModel->all();

        // Haal voor elk product de hoofdfoto op
        foreach ($products as &$product) {
            $primaryImage = $this->productImageModel->getPrimary((int)$product['id']);
            if ($primaryImage) {
                $product['image_url'] = $primaryImage['image_path'];
            } elseif (empty($product['image_url'])) {
                $product['image_url'] = null;
            }
        }
        unset($product); // Break reference

        $this->view('products.index', [
            'user' => $this->user(),
            'title' => 'Producten',
            'products' => $products
        ]);
    }

    /**
     * Nieuw product formulier
     */
    public function create(): void
    {
        $this->view('products.create', [
            'user' => $this->user(),
            'title' => 'Nieuw Product'
        ]);
    }

    /**
     * Product opslaan
     */
    public function store(): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        // Input ophalen
        $data = [
            'product_id' => $_POST['product_id'] ?? '',
            'name' => $_POST['name'] ?? '',
            'sku' => $_POST['sku'] ?? null,
            'description' => $_POST['description'] ?? null,
            'category' => $_POST['category'] ?? null,
            'price' => $_POST['price'] ?? 0,
            'compare_price' => !empty($_POST['compare_price']) ? $_POST['compare_price'] : null,
            'currency' => $_POST['currency'] ?? 'EUR',
            'is_active' => isset($_POST['is_active']) ? 1 : 0
        ];

        // Validatie
        $errors = $this->validateProduct($data);

        if (!empty($errors)) {
            foreach ($errors as $field => $message) {
                $this->flash('error', $message);
            }
            $this->back();
            return;
        }

        // Check of product_id al bestaat
        $existing = $this->productModel->findByProductId($data['product_id']);
        if ($existing) {
            $this->flash('error', 'Product ID bestaat al. Kies een unieke Product ID.');
            $this->back();
            return;
        }

        // Product aanmaken
        $productId = $this->productModel->create($data);

        // File uploads verwerken (meerdere afbeeldingen)
        if ($productId && isset($_FILES['images']) && !empty($_FILES['images']['name'][0])) {
            $uploadedImages = [];
            $uploadErrors = [];

            foreach ($_FILES['images']['name'] as $key => $name) {
                if ($_FILES['images']['error'][$key] === UPLOAD_ERR_OK) {
                    try {
                        $file = [
                            'name' => $_FILES['images']['name'][$key],
                            'type' => $_FILES['images']['type'][$key],
                            'tmp_name' => $_FILES['images']['tmp_name'][$key],
                            'error' => $_FILES['images']['error'][$key],
                            'size' => $_FILES['images']['size'][$key]
                        ];

                        $imagePath = FileUpload::uploadImage($file, 'furniture');
                        if ($imagePath) {
                            $uploadedImages[] = [
                                'path' => $imagePath,
                                'is_primary' => $key === 0, // Eerste afbeelding is primair
                                'sort_order' => $key
                            ];
                        }
                    } catch (\Exception $e) {
                        $uploadErrors[] = $e->getMessage();
                    }
                }
            }

            // Sla alle afbeeldingen op
            foreach ($uploadedImages as $img) {
                $this->productImageModel->addImage(
                    (int)$productId,
                    $img['path'],
                    $img['is_primary'],
                    $img['sort_order']
                );
            }

            // Update ook image_url voor backwards compatibility (eerste afbeelding)
            if (!empty($uploadedImages)) {
                $this->productModel->update($productId, ['image_url' => $uploadedImages[0]['path']]);
            }

            if (!empty($uploadErrors)) {
                $this->flash('warning', 'Product aangemaakt, maar sommige afbeeldingen konden niet worden geüpload: ' . implode(', ', $uploadErrors));
            }
        }

        if ($productId) {
            $this->flash('success', 'Product succesvol toegevoegd.');
            $this->redirect('/products');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het aanmaken van het product.');
            $this->back();
        }
    }

    /**
     * Product bewerken formulier
     */
    public function edit(int $id): void
    {
        $product = $this->productModel->find($id);

        if (!$product) {
            $this->flash('error', 'Product niet gevonden.');
            $this->redirect('/products');
            return;
        }

        // Haal alle afbeeldingen op
        $images = $this->productImageModel->getByProduct($id);
        
        // Fallback: als er geen afbeeldingen in furniture_product_images zijn, maar wel in image_url
        if (empty($images) && !empty($product['image_url'])) {
            // Voeg de oude image_url toe als primaire afbeelding
            $primaryImageId = $this->productImageModel->addImage(
                $id,
                $product['image_url'],
                true,
                0
            );
            
            // Haal opnieuw op
            $images = $this->productImageModel->getByProduct($id);
        }

        $this->view('products.edit', [
            'user' => $this->user(),
            'title' => 'Product Bewerken',
            'product' => $product,
            'images' => $images
        ]);
    }

    /**
     * Product updaten
     */
    public function update(int $id): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        // Product ophalen
        $product = $this->productModel->find($id);

        if (!$product) {
            $this->flash('error', 'Product niet gevonden.');
            $this->redirect('/products');
            return;
        }

        // Input ophalen
        $data = [
            'product_id' => $_POST['product_id'] ?? $product['product_id'],
            'name' => $_POST['name'] ?? '',
            'sku' => $_POST['sku'] ?? null,
            'description' => $_POST['description'] ?? null,
            'category' => $_POST['category'] ?? null,
            'price' => $_POST['price'] ?? 0,
            'compare_price' => !empty($_POST['compare_price']) ? $_POST['compare_price'] : null,
            'currency' => $_POST['currency'] ?? 'EUR',
            'is_active' => isset($_POST['is_active']) ? 1 : 0
        ];

        // Validatie
        $errors = $this->validateProduct($data);

        if (!empty($errors)) {
            foreach ($errors as $field => $message) {
                $this->flash('error', $message);
            }
            $this->back();
            return;
        }

        // Check of product_id al bestaat (bij andere product)
        if ($data['product_id'] !== $product['product_id']) {
            $existing = $this->productModel->findByProductId($data['product_id']);
            if ($existing && $existing['id'] != $id) {
                $this->flash('error', 'Product ID bestaat al. Kies een unieke Product ID.');
                $this->back();
                return;
            }
        }

        // Product updaten
        $updateSuccess = $this->productModel->update($id, $data);

        // File uploads verwerken (nieuwe afbeeldingen)
        if ($updateSuccess && isset($_FILES['images']) && !empty($_FILES['images']['name'][0])) {
            $uploadedImages = [];
            $uploadErrors = [];

            // Haal huidige afbeeldingen op om sort_order te bepalen
            $existingImages = $this->productImageModel->getByProduct($id);
            $nextSortOrder = count($existingImages);
            
            // Check if user selected a specific new image to be primary
            $newPrimaryIndex = isset($_POST['new_primary_index']) && $_POST['new_primary_index'] !== '' 
                ? (int)$_POST['new_primary_index'] 
                : -1;
            
            error_log("New primary index from form: " . $newPrimaryIndex);

            foreach ($_FILES['images']['name'] as $key => $name) {
                if ($_FILES['images']['error'][$key] === UPLOAD_ERR_OK) {
                    try {
                        $file = [
                            'name' => $_FILES['images']['name'][$key],
                            'type' => $_FILES['images']['type'][$key],
                            'tmp_name' => $_FILES['images']['tmp_name'][$key],
                            'error' => $_FILES['images']['error'][$key],
                            'size' => $_FILES['images']['size'][$key]
                        ];

                        $imagePath = FileUpload::uploadImage($file, 'furniture');
                        if ($imagePath) {
                            // Check if THIS new image should be primary
                            $shouldBePrimary = ($newPrimaryIndex === $key);
                            
                            $uploadedImages[] = [
                                'path' => $imagePath,
                                'is_primary' => $shouldBePrimary,
                                'sort_order' => $nextSortOrder + $key
                            ];
                            
                            error_log("Uploaded image $key: is_primary = " . ($shouldBePrimary ? 'true' : 'false'));
                        }
                    } catch (\Exception $e) {
                        $uploadErrors[] = $e->getMessage();
                    }
                }
            }

            // Als er geen bestaande primaire afbeelding is EN geen nieuwe primary is geselecteerd, 
            // maak de eerste nieuwe primair
            $hasPrimary = false;
            foreach ($existingImages as $img) {
                if ($img['is_primary']) {
                    $hasPrimary = true;
                    break;
                }
            }
            
            $hasNewPrimary = false;
            foreach ($uploadedImages as $img) {
                if ($img['is_primary']) {
                    $hasNewPrimary = true;
                    break;
                }
            }
            
            if (!$hasPrimary && !$hasNewPrimary && !empty($uploadedImages)) {
                $uploadedImages[0]['is_primary'] = true;
                error_log("No primary found, making first new image primary");
            }
            
            // If a new image is set as primary, remove primary from existing images
            if ($hasNewPrimary) {
                error_log("New image is primary, removing primary from existing images");
                foreach ($existingImages as $existingImg) {
                    if ($existingImg['is_primary']) {
                        $this->productImageModel->setPrimary($existingImg['id'], false);
                    }
                }
            }

            // Sla nieuwe afbeeldingen op
            foreach ($uploadedImages as $img) {
                $this->productImageModel->addImage(
                    $id,
                    $img['path'],
                    $img['is_primary'],
                    $img['sort_order']
                );
            }

            // Update image_url voor backwards compatibility (primaire afbeelding)
            $primaryImage = $this->productImageModel->getPrimary($id);
            if ($primaryImage) {
                $this->productModel->update($id, ['image_url' => $primaryImage['image_path']]);
            }

            if (!empty($uploadErrors)) {
                $this->flash('warning', 'Product bijgewerkt, maar sommige afbeeldingen konden niet worden geüpload: ' . implode(', ', $uploadErrors));
            }
        }

        if ($updateSuccess) {
            $this->flash('success', 'Product succesvol bijgewerkt.');
            $this->redirect('/products');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het bijwerken van het product.');
            $this->back();
        }
    }

    /**
     * Product verwijderen
     */
    public function delete(int $id): void
    {
        if (!validate_csrf()) {
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->redirect('/products');
            return;
        }

        $product = $this->productModel->find($id);
        if (!$product) {
            $this->flash('error', 'Product niet gevonden.');
            $this->redirect('/products');
            return;
        }

        if ($this->productModel->delete($id)) {
            $this->flash('success', 'Product succesvol verwijderd.');
        } else {
            $this->flash('error', 'Er is een fout opgetreden bij het verwijderen van het product.');
        }

        $this->redirect('/products');
    }

    /**
     * Stel primaire afbeelding in
     */
    public function setPrimaryImage(int $imageId): void
    {
        // Check if AJAX request
        $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                  strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
        
        if (!validate_csrf()) {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Ongeldige aanvraag.']);
                exit;
            }
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        $image = $this->productImageModel->find($imageId);
        if (!$image) {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Afbeelding niet gevonden.']);
                exit;
            }
            $this->flash('error', 'Afbeelding niet gevonden.');
            $this->back();
            return;
        }

        if ($this->productImageModel->setPrimary($imageId, (int)$image['product_id'])) {
            // Update ook image_url voor backwards compatibility
            $this->productModel->update((int)$image['product_id'], ['image_url' => $image['image_path']]);
            
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => true, 
                    'message' => 'Hoofdfoto bijgewerkt.',
                    'imageId' => $imageId
                ]);
                exit;
            }
            $this->flash('success', 'Hoofdfoto bijgewerkt.');
        } else {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Er is een fout opgetreden.']);
                exit;
            }
            $this->flash('error', 'Er is een fout opgetreden.');
        }

        $this->redirect('/products/edit/' . $image['product_id']);
    }

    /**
     * Verwijder afbeelding
     */
    public function deleteImage(int $imageId): void
    {
        // Check if AJAX request
        $isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
                  strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
        
        if (!validate_csrf()) {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Ongeldige aanvraag.']);
                exit;
            }
            $this->flash('error', 'Ongeldige aanvraag.');
            $this->back();
            return;
        }

        $image = $this->productImageModel->find($imageId);
        if (!$image) {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Afbeelding niet gevonden.']);
                exit;
            }
            $this->flash('error', 'Afbeelding niet gevonden.');
            $this->back();
            return;
        }

        $productId = (int)$image['product_id'];
        $wasPrimary = (bool)$image['is_primary'];
        $imagePath = $image['image_path'];
        $remainingImages = [];

        // Verwijder fysiek bestand eerst
        if (!empty($imagePath)) {
            FileUpload::deleteImage($imagePath);
        }

        if ($this->productImageModel->removeImage($imageId)) {
            // Als dit de primaire was, maak de eerste overgebleven afbeelding primair
            if ($wasPrimary) {
                $remainingImages = $this->productImageModel->getByProduct($productId);
                if (!empty($remainingImages)) {
                    $this->productImageModel->setPrimary((int)$remainingImages[0]['id'], $productId);
                    // Update image_url
                    $this->productModel->update($productId, ['image_url' => $remainingImages[0]['image_path']]);
                } else {
                    // Geen afbeeldingen meer, zet image_url op null
                    $this->productModel->update($productId, ['image_url' => null]);
                }
            }
            
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => true, 
                    'message' => 'Afbeelding verwijderd.',
                    'newPrimaryImageId' => !empty($remainingImages) ? (int)$remainingImages[0]['id'] : null
                ]);
                exit;
            }
            $this->flash('success', 'Afbeelding verwijderd.');
        } else {
            if ($isAjax) {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'message' => 'Er is een fout opgetreden.']);
                exit;
            }
            $this->flash('error', 'Er is een fout opgetreden.');
        }

        $this->redirect('/products/edit/' . $productId);
    }

    /**
     * Valideer product data
     */
    private function validateProduct(array $data): array
    {
        $errors = [];

        if (empty($data['product_id'])) {
            $errors['product_id'] = 'Product ID is verplicht.';
        } elseif (!preg_match('/^[A-Z0-9\-_]+$/', $data['product_id'])) {
            $errors['product_id'] = 'Product ID mag alleen hoofdletters, cijfers, streepjes en underscores bevatten.';
        }

        if (empty($data['name'])) {
            $errors['name'] = 'Productnaam is verplicht.';
        }

        if (empty($data['price']) || !is_numeric($data['price']) || $data['price'] < 0) {
            $errors['price'] = 'Voer een geldige prijs in (0 of hoger).';
        }

        if (!empty($data['compare_price']) && (!is_numeric($data['compare_price']) || $data['compare_price'] < 0)) {
            $errors['compare_price'] = 'Vergelijkingsprijs moet een geldig getal zijn (0 of hoger).';
        }

        if (!empty($data['compare_price']) && !empty($data['price']) && $data['compare_price'] <= $data['price']) {
            $errors['compare_price'] = 'Vergelijkingsprijs moet hoger zijn dan de normale prijs.';
        }

        if (!empty($data['currency']) && strlen($data['currency']) !== 3) {
            $errors['currency'] = 'Currency moet 3 karakters zijn (bijv. EUR, USD).';
        }

        return $errors;
    }
}
