<?php
namespace App\Services;

use App\Models\ApiCredential;
use App\Models\FurnitureOrder;
use Core\Database;

/**
 * WebhookService
 * Stuurt payment updates naar VPS1 via webhooks
 */
class WebhookService
{
    private ApiCredential $apiCredentialModel;
    private FurnitureOrder $orderModel;
    private Database $db;
    
    private int $timeout = 30;
    private int $maxRetries = 3;
    
    public function __construct()
    {
        $this->apiCredentialModel = new ApiCredential();
        $this->orderModel = new FurnitureOrder();
        $this->db = Database::getInstance();
    }

    /**
     * Stuur order created webhook naar VPS1
     * Wordt aangeroepen wanneer een nieuwe order wordt aangemaakt op VPS2
     */
    public function sendOrderCreated(string $furnitureOrderId): bool
    {
        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($furnitureOrderId);
        
        if (!$order || !$order['webhook_url']) {
            error_log("WebhookService: Order niet gevonden of geen webhook_url voor {$furnitureOrderId}");
            return false;
        }

        // Haal API credentials op voor webhook secret
        $credentials = $this->apiCredentialModel->getActiveCredentials();
        
        if (!$credentials || !$credentials['webhook_secret']) {
            error_log("WebhookService: Geen actieve credentials of webhook_secret");
            return false;
        }

        // Haal webhook secret op
        $webhookSecret = $credentials['webhook_secret'];

        // Bouw webhook payload voor order.created event
        $payload = [
            'event' => 'order.created',
            'furniture_order_id' => $furnitureOrderId,
            'external_order_id' => $order['external_order_id'],
            'product_id' => $order['product_id'],
            'product_name' => $order['product_name'],
            'quantity' => (int) $order['quantity'],
            'price' => (float) $order['price'],
            'currency' => $order['currency'] ?? 'EUR',
            'payment_status' => $order['payment_status'] ?? 'pending',
            'payment_url' => $order['payment_url'] ?? null,
            'payment_url_expires_at' => $order['payment_url_expires_at'] ?? null,
            'created_at' => $order['created_at']
        ];

        // Genereer HMAC signature
        $payloadJson = json_encode($payload);
        $signature = hash_hmac('sha256', $payloadJson, $webhookSecret);

        // Stuur webhook met retry logic
        $success = $this->sendWebhook($order['webhook_url'], $payloadJson, $signature, $webhookSecret);

        // Markeer webhook als verstuurd indien succesvol
        if ($success) {
            $this->db->execute(
                "UPDATE furniture_orders SET webhook_sent_at = NOW() WHERE furniture_order_id = ?",
                [$furnitureOrderId]
            );
        }

        return $success;
    }

    /**
     * Stuur payment status update naar VPS1
     */
    public function sendPaymentStatusUpdate(string $furnitureOrderId, string $status, array $data = []): bool
    {
        // Haal order op
        $order = $this->orderModel->findByFurnitureOrderId($furnitureOrderId);
        
        if (!$order || !$order['webhook_url']) {
            error_log("WebhookService: Order niet gevonden of geen webhook_url voor {$furnitureOrderId}");
            return false;
        }

        // Haal API credentials op voor webhook secret
        $credentials = $this->apiCredentialModel->getActiveCredentials();
        
        if (!$credentials || !$credentials['webhook_secret']) {
            error_log("WebhookService: Geen actieve credentials of webhook_secret");
            return false;
        }

        // Haal webhook secret op
        $webhookSecret = $credentials['webhook_secret'];

        // Bouw webhook payload (volgens specificaties)
        $payload = [
            'event' => 'payment.' . $status,
            'furniture_order_id' => $furnitureOrderId,
            'external_order_id' => $order['external_order_id'],
            'payment_status' => $status,
            'amount' => (float) $order['price'],
            'currency' => $order['currency']
        ];

        // Voeg extra data toe
        if (isset($data['transaction_id'])) {
            $payload['transaction_id'] = $data['transaction_id'];
        }
        
        if (isset($data['paid_at'])) {
            $payload['paid_at'] = $data['paid_at'];
        } elseif ($order['paid_at']) {
            $payload['paid_at'] = $order['paid_at'];
        }

        if (isset($data['failure_reason'])) {
            $payload['failure_reason'] = $data['failure_reason'];
        }

        // Genereer HMAC signature
        $payloadJson = json_encode($payload);
        $signature = hash_hmac('sha256', $payloadJson, $webhookSecret);

        // Stuur webhook met retry logic
        $success = $this->sendWebhook($order['webhook_url'], $payloadJson, $signature, $webhookSecret);

        // Markeer webhook als verstuurd
        if ($success) {
            $this->orderModel->markWebhookSent($furnitureOrderId);
        }

        return $success;
    }

    /**
     * Stuur webhook met retry logic
     */
    private function sendWebhook(string $url, string $payload, string $signature, string $secret, int $retryCount = 0): bool
    {
        // URL-encode de URL om spaties te escapen (Dashboard 1 heeft spatie in naam)
        $url = str_replace(' ', '%20', $url);
        
        $ch = curl_init($url);
        
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-Key: ' . $secret, // Webhook secret als API key
                'X-Signature: ' . $signature
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        // Log webhook call
        $this->logWebhookCall($url, $payload, $httpCode, $response, $error, $retryCount);

        // Success (2xx status codes)
        if ($httpCode >= 200 && $httpCode < 300) {
            return true;
        }

        // Retry bij failure (max 3x)
        if ($retryCount < $this->maxRetries && ($httpCode >= 500 || $httpCode === 0)) {
            sleep(pow(2, $retryCount)); // Exponential backoff
            return $this->sendWebhook($url, $payload, $signature, $secret, $retryCount + 1);
        }

        return false;
    }

    /**
     * Log webhook call
     */
    private function logWebhookCall(string $url, string $payload, int $httpCode, ?string $response, ?string $error, int $retryCount): void
    {
        $this->db->execute(
            "INSERT INTO api_access_log 
             (endpoint, ip_address, request_method, request_payload, response_status, response_payload, error_message, created_at)
             VALUES (?, ?, ?, ?, ?, ?, ?, NOW())",
            [
                $url,
                'outgoing',
                'POST',
                $payload,
                $httpCode,
                $response,
                $error ? "Retry: {$retryCount}, Error: {$error}" : null
            ]
        );
    }

    /**
     * Verwerk alle pending webhooks (voor cron job)
     */
    public function processPendingWebhooks(): int
    {
        $orders = $this->orderModel->getPendingWebhooks();
        $sent = 0;

        foreach ($orders as $order) {
            $success = $this->sendPaymentStatusUpdate(
                $order['furniture_order_id'],
                $order['payment_status'],
                [
                    'transaction_id' => $order['transaction_id'],
                    'paid_at' => $order['paid_at'],
                    'failure_reason' => $order['failure_reason']
                ]
            );

            if ($success) {
                $sent++;
            }
        }

        return $sent;
    }
}

