<?php
namespace Core;

/**
 * Request Class
 * Beheert alle HTTP request data
 */
class Request
{
    private array $get;
    private array $post;
    private array $server;
    private array $files;
    private array $cookies;

    public function __construct()
    {
        $this->get = $_GET;
        $this->post = $_POST;
        $this->server = $_SERVER;
        $this->files = $_FILES;
        $this->cookies = $_COOKIE;
    }

    /**
     * HTTP method ophalen
     */
    public function method(): string
    {
        return strtoupper($this->server['REQUEST_METHOD'] ?? 'GET');
    }

    /**
     * URI ophalen
     */
    public function uri(): string
    {
        // Eerst kijken naar de 'url' parameter (van .htaccess)
        if (isset($_GET['url'])) {
            $uri = '/' . trim($_GET['url'], '/');
            return $uri !== '/' ? $uri : '/';
        }
        
        // Anders REQUEST_URI gebruiken en query string verwijderen
        $uri = $this->server['REQUEST_URI'] ?? '/';
        $uri = parse_url($uri, PHP_URL_PATH);
        
        // Verwijder base path (bijv. /Oraviare/Admin/public)
        $scriptName = dirname($this->server['SCRIPT_NAME'] ?? '/');
        $scriptName = str_replace('\\', '/', $scriptName); // Normaliseer voor Windows
        
        if ($scriptName !== '/' && $scriptName !== '\\' && $scriptName !== '.' && strpos($uri, $scriptName) === 0) {
            $uri = substr($uri, strlen($scriptName));
        }
        
        // Normaliseer naar root als leeg
        $uri = '/' . trim($uri, '/');
        return $uri === '/' ? '/' : $uri;
    }

    /**
     * GET parameter ophalen
     */
    public function get(string $key, $default = null)
    {
        return $this->get[$key] ?? $default;
    }

    /**
     * POST parameter ophalen
     */
    public function post(string $key, $default = null)
    {
        return $this->post[$key] ?? $default;
    }

    /**
     * Alle POST data ophalen
     */
    public function all(): array
    {
        return array_merge($this->get, $this->post);
    }

    /**
     * Alleen specifieke velden ophalen
     */
    public function only(array $keys): array
    {
        $data = [];
        foreach ($keys as $key) {
            if (isset($this->post[$key])) {
                $data[$key] = $this->post[$key];
            } elseif (isset($this->get[$key])) {
                $data[$key] = $this->get[$key];
            }
        }
        return $data;
    }

    /**
     * Check of request via AJAX is
     */
    public function isAjax(): bool
    {
        return isset($this->server['HTTP_X_REQUESTED_WITH']) 
            && strtolower($this->server['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
    }

    /**
     * Check of request via POST is
     */
    public function isPost(): bool
    {
        return $this->method() === 'POST';
    }

    /**
     * Check of request via GET is
     */
    public function isGet(): bool
    {
        return $this->method() === 'GET';
    }

    /**
     * Input sanitizen
     */
    public function sanitize(string $key, $default = null)
    {
        $value = $this->post($key, $this->get($key, $default));
        
        if (is_string($value)) {
            return htmlspecialchars(trim($value), ENT_QUOTES, 'UTF-8');
        }
        
        return $value;
    }

    /**
     * Valideer CSRF token
     */
    public function validateCsrfToken(): bool
    {
        $token = $this->post('csrf_token');
        return $token && isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }

    /**
     * File upload ophalen
     */
    public function file(string $key): ?array
    {
        return $this->files[$key] ?? null;
    }

    /**
     * Check of er een file upload is
     */
    public function hasFile(string $key): bool
    {
        return isset($this->files[$key]) && $this->files[$key]['error'] === UPLOAD_ERR_OK;
    }
}

