<?php
namespace Core;

/**
 * Router Class
 * Beheert alle routes en dispatcht requests naar de juiste controllers
 */
class Router
{
    public array $routes = [];
    private array $middlewares = [];

    /**
     * GET route registreren
     */
    public function get(string $uri, $callback): void
    {
        $this->routes['GET'][$uri] = $callback;
    }

    /**
     * POST route registreren
     */
    public function post(string $uri, $callback): void
    {
        $this->routes['POST'][$uri] = $callback;
    }

    /**
     * PUT route registreren
     */
    public function put(string $uri, $callback): void
    {
        $this->routes['PUT'][$uri] = $callback;
    }

    /**
     * DELETE route registreren
     */
    public function delete(string $uri, $callback): void
    {
        $this->routes['DELETE'][$uri] = $callback;
    }

    /**
     * Middleware toevoegen
     */
    public function middleware(string $name, callable $callback): void
    {
        $this->middlewares[$name] = $callback;
    }

    /**
     * Request dispatchen naar juiste controller
     */
    public function dispatch(Request $request): void
    {
        $method = $request->method();
        $uri = $this->parseUri($request->uri());

        // Route zoeken
        if (!isset($this->routes[$method])) {
            $this->notFound();
            return;
        }

        foreach ($this->routes[$method] as $route => $callback) {
            $pattern = $this->convertRouteToRegex($route);
            
            if (preg_match($pattern, $uri, $matches)) {
                array_shift($matches); // Verwijder de volledige match
                
                // Callback uitvoeren
                if (is_array($callback)) {
                    $this->handleControllerCallback($callback, $matches);
                } else {
                    call_user_func_array($callback, $matches);
                }
                return;
            }
        }

        $this->notFound();
    }

    /**
     * URI parsen en schoonmaken
     */
    private function parseUri(string $uri): string
    {
        $uri = parse_url($uri, PHP_URL_PATH);
        return '/' . trim($uri, '/');
    }

    /**
     * Route pattern converteren naar regex
     */
    private function convertRouteToRegex(string $route): string
    {
        $route = '/' . trim($route, '/');
        // Accepteer alfanumerieke tekens, koppeltekens en underscores voor parameters (voor slugs)
        $route = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '([a-zA-Z0-9_-]+)', $route);
        return '#^' . $route . '$#';
    }

    /**
     * Controller callback afhandelen
     */
    private function handleControllerCallback(array $callback, array $params): void
    {
        [$controller, $method] = $callback;
        
        // Controller instantiëren
        if (!class_exists($controller)) {
            die("Controller niet gevonden: {$controller}");
        }
        
        $controllerInstance = new $controller();
        
        if (!method_exists($controllerInstance, $method)) {
            die("Method niet gevonden: {$method} in {$controller}");
        }
        
        call_user_func_array([$controllerInstance, $method], $params);
    }

    /**
     * 404 Not Found
     */
    private function notFound(): void
    {
        http_response_code(404);
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'error' => 'Not Found',
            'message' => 'Endpoint not found'
        ]);
        exit;
    }
}

