<?php
/**
 * Helper Functions
 * Globale utility functies beschikbaar in de hele applicatie
 */

/**
 * Debug functie (alleen in development)
 */
function dd(...$vars): void
{
    echo '<pre style="background: #1e1e1e; color: #00ff00; padding: 20px; border-radius: 5px; font-family: monospace;">';
    foreach ($vars as $var) {
        var_dump($var);
    }
    echo '</pre>';
    die();
}

/**
 * View renderen
 */
function view(string $view, array $data = []): void
{
    extract($data);
    $viewFile = __DIR__ . '/../app/Views/' . str_replace('.', '/', $view) . '.php';
    
    if (file_exists($viewFile)) {
        require_once $viewFile;
    } else {
        die("View niet gevonden: {$view}");
    }
}

/**
 * Redirect functie
 */
function redirect(string $path): void
{
    $config = require __DIR__ . '/../config/app.php';
    $baseUrl = rtrim($config['url'], '/');
    header("Location: {$baseUrl}{$path}");
    exit;
}

/**
 * XSS protection voor output
 */
function e($value): string
{
    if ($value === null) {
        return '';
    }
    return htmlspecialchars((string) $value, ENT_QUOTES, 'UTF-8');
}

/**
 * URL genereren
 */
function url(string $path = ''): string
{
    $config = require __DIR__ . '/../config/app.php';
    return rtrim($config['url'], '/') . '/' . ltrim($path, '/');
}

/**
 * Asset URL genereren
 */
function asset(string $path): string
{
    return url('assets/' . ltrim($path, '/'));
}

/**
 * Check of gebruiker is ingelogd
 */
function auth(): ?array
{
    if (!isset($_SESSION['user_id'])) {
        return null;
    }
    
    // Gebruiker data ophalen (cache in session)
    if (!isset($_SESSION['user_data'])) {
        $db = \Core\Database::getInstance();
        $_SESSION['user_data'] = $db->queryOne(
            "SELECT id, username, email, role FROM users WHERE id = ?",
            [$_SESSION['user_id']]
        );
    }
    
    return $_SESSION['user_data'];
}

/**
 * Flash message instellen of ophalen
 */
function flash(string $key, $value = null)
{
    if ($value === null) {
        $message = $_SESSION['flash'][$key] ?? null;
        unset($_SESSION['flash'][$key]);
        return $message;
    }
    $_SESSION['flash'][$key] = $value;
}

/**
 * Old input value ophalen (na form submission)
 */
function old(string $key, $default = '')
{
    $value = $_SESSION['old'][$key] ?? $default;
    unset($_SESSION['old'][$key]);
    return $value;
}

/**
 * Errors ophalen (na form validation)
 */
function errors(string $key = null)
{
    if ($key === null) {
        $errors = $_SESSION['errors'] ?? [];
        unset($_SESSION['errors']);
        return $errors;
    }
    
    $error = $_SESSION['errors'][$key] ?? null;
    return $error;
}

/**
 * CSRF token genereren
 */
function csrf_token(): string
{
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        $_SESSION['csrf_token_time'] = time();
    }
    return $_SESSION['csrf_token'];
}

/**
 * CSRF token field voor forms
 */
function csrf_field(): string
{
    return '<input type="hidden" name="csrf_token" value="' . csrf_token() . '">';
}

/**
 * Valideer CSRF token
 */
function validate_csrf(): bool
{
    $token = $_POST['csrf_token'] ?? '';
    
    // Als er geen token in POST staat maar wel in GET (fallback)
    if (empty($token) && isset($_GET['csrf_token'])) {
        $token = $_GET['csrf_token'];
    }
    
    return isset($_SESSION['csrf_token']) && !empty($token) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Format datum
 */
function format_date($date, string $format = 'd-m-Y H:i'): string
{
    if (empty($date)) {
        return '';
    }
    
    if (is_string($date)) {
        $date = new DateTime($date);
    }
    
    return $date->format($format);
}

/**
 * Format prijs (in euro's)
 */
function format_price($amount): string
{
    return '€ ' . number_format((float) $amount, 2, ',', '.');
}

/**
 * Genereer random string
 */
function str_random(int $length = 16): string
{
    return bin2hex(random_bytes($length / 2));
}

/**
 * Password hashen
 */
function hash_password(string $password): string
{
    return password_hash($password, PASSWORD_DEFAULT);
}

/**
 * Password verifiëren
 */
function verify_password(string $password, string $hash): bool
{
    return password_verify($password, $hash);
}

/**
 * Check of string een geldig email adres is
 */
function is_email(string $email): bool
{
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Configuratie waarde ophalen
 */
function config(string $key, $default = null)
{
    static $config = null;
    
    if ($config === null) {
        $config = [
            'app' => require __DIR__ . '/../config/app.php',
            'database' => require __DIR__ . '/../config/database.php',
        ];
    }
    
    $keys = explode('.', $key);
    $value = $config;
    
    foreach ($keys as $k) {
        if (!isset($value[$k])) {
            return $default;
        }
        $value = $value[$k];
    }
    
    return $value;
}

