-- =============================================
-- VPS2 FURNITURE SHOP - DATABASE SCHEMA
-- Meubelwebshop voor payment processing
-- =============================================

CREATE DATABASE IF NOT EXISTS `oraviare` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE `oraviare`;

-- =============================================
-- TABEL 1: furniture_orders
-- Orders van VPS1 die hier verwerkt worden
-- =============================================
CREATE TABLE IF NOT EXISTS `furniture_orders` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `furniture_order_id` VARCHAR(100) NOT NULL UNIQUE COMMENT 'FURN-67890',
    `external_order_id` VARCHAR(100) NOT NULL COMMENT 'VAPE-12345 van VPS1',
    `product_id` VARCHAR(100) NOT NULL COMMENT 'STOEL-WHITE-BEIGE',
    `product_name` VARCHAR(255) NOT NULL COMMENT 'Witte Design Stoel',
    `quantity` INT NOT NULL DEFAULT 1,
    `price` DECIMAL(10, 2) NOT NULL,
    `currency` VARCHAR(3) DEFAULT 'EUR',
    `payment_status` ENUM('pending', 'paid', 'failed', 'cancelled') DEFAULT 'pending',
    `payment_url` TEXT NULL COMMENT 'Checkout URL voor payment provider',
    `payment_url_expires_at` DATETIME NULL,
    `transaction_id` VARCHAR(100) NULL COMMENT 'Van payment provider',
    `paid_at` DATETIME NULL,
    `failed_at` DATETIME NULL,
    `failure_reason` TEXT NULL,
    `webhook_sent` BOOLEAN DEFAULT 0 COMMENT 'Webhook naar VPS1 verstuurd?',
    `webhook_sent_at` DATETIME NULL,
    `customer_email` VARCHAR(255) NULL COMMENT 'Placeholder email van VPS1',
    `customer_name` VARCHAR(255) NULL COMMENT 'Placeholder naam van VPS1',
    `return_url` TEXT NULL COMMENT 'Return URL naar VPS1',
    `cancel_url` TEXT NULL COMMENT 'Cancel URL naar VPS1',
    `webhook_url` TEXT NULL COMMENT 'Webhook URL naar VPS1',
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_furniture_order_id` (`furniture_order_id`),
    INDEX `idx_external_order` (`external_order_id`),
    INDEX `idx_payment_status` (`payment_status`),
    INDEX `idx_webhook_sent` (`webhook_sent`),
    INDEX `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABEL 2: api_access_log
-- Logging van alle API calls van VPS1
-- =============================================
CREATE TABLE IF NOT EXISTS `api_access_log` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `api_key_hash` VARCHAR(64) NULL COMMENT 'Hash van API key voor logging',
    `endpoint` VARCHAR(255) NOT NULL,
    `ip_address` VARCHAR(45) NOT NULL,
    `request_method` VARCHAR(10) NOT NULL,
    `request_payload` TEXT NULL COMMENT 'JSON',
    `response_status` INT NULL,
    `response_payload` TEXT NULL COMMENT 'JSON',
    `error_message` TEXT NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_ip` (`ip_address`),
    INDEX `idx_endpoint` (`endpoint`),
    INDEX `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABEL 3: furniture_products
-- Furniture producten die gekoppeld worden aan VPS1 producten
-- =============================================
CREATE TABLE IF NOT EXISTS `furniture_products` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `product_id` VARCHAR(100) NOT NULL UNIQUE COMMENT 'STOEL-WHITE-BEIGE',
    `name` VARCHAR(255) NOT NULL COMMENT 'Witte Design Stoel met Beige Accenten',
    `sku` VARCHAR(100) NULL,
    `description` TEXT NULL,
    `image_url` VARCHAR(255) NULL COMMENT 'Hoofdafbeelding van het product',
    `category` VARCHAR(100) NULL COMMENT 'Categorie van het product',
    `price` DECIMAL(10, 2) NOT NULL,
    `compare_price` DECIMAL(10, 2) NULL COMMENT 'Vergelijkingsprijs (originele prijs)',
    `currency` VARCHAR(3) DEFAULT 'EUR',
    `is_active` BOOLEAN DEFAULT 1,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_product_id` (`product_id`),
    INDEX `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABEL 4: api_credentials
-- API keys voor VPS1 authenticatie
-- =============================================
CREATE TABLE IF NOT EXISTS `api_credentials` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `name` VARCHAR(100) NOT NULL COMMENT 'VPS1 API Key',
    `api_key` VARCHAR(255) NOT NULL COMMENT 'API key voor authenticatie',
    `api_secret` VARCHAR(255) NOT NULL COMMENT 'Secret voor HMAC signing',
    `webhook_secret` VARCHAR(255) NULL COMMENT 'Secret voor webhook verificatie',
    `vps1_url` VARCHAR(255) NULL COMMENT 'URL van VPS1 voor webhooks',
    `is_active` BOOLEAN DEFAULT 1,
    `allowed_ips` TEXT NULL COMMENT 'JSON array met toegestane IP adressen',
    `last_used_at` DATETIME NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_api_key` (`api_key`),
    INDEX `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABEL 5: furniture_categories
-- Categorieën voor furniture producten
-- =============================================
CREATE TABLE IF NOT EXISTS `furniture_categories` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `name` VARCHAR(100) NOT NULL UNIQUE,
    `slug` VARCHAR(100) NOT NULL UNIQUE,
    `description` TEXT NULL,
    `sort_order` INT DEFAULT 0,
    `status` ENUM('active', 'inactive') DEFAULT 'active',
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_name` (`name`),
    UNIQUE KEY `uk_slug` (`slug`),
    INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- DEMO DATA: Furniture Categories
-- =============================================
INSERT INTO `furniture_categories` (`name`, `slug`, `description`, `sort_order`, `status`) VALUES
('Stoelen', 'stoelen', 'Alle stoelen en zitmeubelen', 1, 'active'),
('Tafels', 'tafels', 'Tafels voor elke ruimte', 2, 'active'),
('Lampen', 'lampen', 'Verlichting en lampen', 3, 'active'),
('Kussens', 'kussens', 'Decoratieve kussens', 4, 'active'),
('Vazen', 'vazen', 'Decoratieve vazen', 5, 'active'),
('Decoratie', 'decoratie', 'Algemene decoratie', 6, 'active'),
('Accessoires', 'accessoires', 'Meubel accessoires', 7, 'active');

-- =============================================
-- DEMO DATA: Furniture Products (voorbeeld)
-- =============================================
INSERT INTO `furniture_products` (`product_id`, `name`, `sku`, `description`, `category`, `price`, `currency`, `is_active`) VALUES
('STOEL-001', 'Witte Stoel Beige Accenten', 'STOEL-WHITE-BEIGE', 'Designer stoel met moderne uitstraling', 'Stoelen', 9.99, 'EUR', 1),
('STOEL-002', 'Grijze Stoel Modern', 'STOEL-GRAY-MODERN', 'Hedendaagse stoel in grijstinten', 'Stoelen', 9.99, 'EUR', 1),
('LAMP-001', 'Design Tafellamp', 'LAMP-DESIGN-TABLE', 'Elegante tafellamp voor moderne interieurs', 'Lampen', 12.99, 'EUR', 1),
('KUSSEN-001', 'Sierkussen Set', 'KUSSEN-DECOR-SET', 'Set van decoratieve kussens', 'Kussens', 14.99, 'EUR', 1),
('VAAS-001', 'Keramische Vaas', 'VAAS-CERAMIC', 'Handgemaakte keramische vaas', 'Vazen', 12.99, 'EUR', 1);

-- =============================================
-- DEMO DATA: API Credentials (placeholder)
-- =============================================
INSERT INTO `api_credentials` (`name`, `api_key`, `api_secret`, `webhook_secret`, `is_active`) VALUES
('VPS1 API Key', 'VPS1_API_KEY_PLACEHOLDER', 'VPS1_API_SECRET_PLACEHOLDER', 'WEBHOOK_SECRET_PLACEHOLDER', 1);

-- =============================================
-- TABEL 6: furniture_product_images
-- Meerdere afbeeldingen per product
-- =============================================
CREATE TABLE IF NOT EXISTS `furniture_product_images` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `product_id` INT UNSIGNED NOT NULL COMMENT 'Foreign key naar furniture_products.id',
    `image_path` VARCHAR(255) NOT NULL COMMENT 'Pad naar afbeelding',
    `is_primary` BOOLEAN DEFAULT 0 COMMENT 'Is dit de hoofdfoto?',
    `sort_order` INT DEFAULT 0 COMMENT 'Sorteervolgorde',
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    INDEX `idx_product` (`product_id`),
    INDEX `idx_primary` (`is_primary`),
    INDEX `idx_sort` (`sort_order`),
    CONSTRAINT `fk_product_images_product` FOREIGN KEY (`product_id`) REFERENCES `furniture_products` (`id`) ON DELETE CASCADE ON UPDATE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABEL 7: users
-- Admin gebruikers voor dashboard (optioneel)
-- =============================================
CREATE TABLE IF NOT EXISTS `users` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
    `username` VARCHAR(50) NOT NULL UNIQUE,
    `email` VARCHAR(255) NOT NULL UNIQUE,
    `password` VARCHAR(255) NOT NULL,
    `role` VARCHAR(20) DEFAULT 'admin',
    `last_login` DATETIME NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    PRIMARY KEY (`id`),
    UNIQUE KEY `uk_username` (`username`),
    UNIQUE KEY `uk_email` (`email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- DEMO DATA: Admin User
-- =============================================
-- Standaard admin: admin / admin123
INSERT INTO `users` (`username`, `email`, `password`, `role`) VALUES
('admin', 'admin@vps2.local', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin');
-- Wachtwoord: admin123

-- =============================================
-- KLAAR! 
-- Database schema voor VPS2 is aangemaakt
-- =============================================

